/**
 * Use of the WebDLPIndexer and related source code is subject to the
 * terms of the following license:
 *
 * Copyright (c) 2013 Carnegie Mellon University. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following acknowledgments and disclaimers.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. All advertising materials for third-party software mentioning features or
 * use of this software must display the following disclaimer:
 *
 * “Neither Carnegie Mellon University nor its Software Engineering Institute
 * have reviewed or endorsed this software”
 *
 * 4. The names “Carnegie Mellon University,” "CERT” and/or “Software
 * Engineering Institute" shall not be used to endorse or promote products
 * derived from this software without prior written permission. For written
 * permission, please contact permission@sei.cmu.edu.
 *
 * 5. Products derived from this software may not be called "CERT" nor may
 * "CERT" appear in their names without prior written permission of
 * permission@sei.cmu.edu.
 *
 * 6. Redistributions of any form whatsoever must retain the following
 * acknowledgment:
 *
 * "This product includes software developed by CERT with funding and support
 * from the US Government under Contract No. FA8721-05-C-0003. The U.S.
 * Government's rights to use, modify, reproduce, release, perform, display, or
 * disclose this material are restricted by the Rights in Technical
 * Data-Noncommercial Items clauses (DFAR 252-227.7013 and DFAR 252-227.7013
 * Alternate I contained in the foregoing identified contract.
 *
 * THIS SOFTWARE IS PROVIDED BY CARNEGIE MELLON UNIVERSITY ``AS IS'' AND
 * CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESS OR
 * IMPLIED, AS TO ANY MATTER, AND ALL SUCH WARRANTIES, INCLUDING WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, ARE EXPRESSLY
 * DISCLAIMED. WITHOUT LIMITING THE GENERALITY OF THE FOREGOING, CARNEGIE MELLON
 * UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND RELATING TO EXCLUSIVITY,
 * INFORMATIONAL CONTENT, ERROR-FREE OPERATION, RESULTS TO BE OBTAINED FROM USE,
 * FREEDOM FROM PATENT, TRADEMARK AND COPYRIGHT INFRINGEMENT AND/OR FREEDOM FROM
 * THEFT OF TRADE SECRETS.”
 *
 */
package webdlpindexer;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.net.InetAddress;
import java.util.Date;

/**
 * A collection of static utility methods for writing messages and errors to a
 * log file or stream.
 *
 * @author Daniel Costa
 */
public class Logger {

    public static boolean logToConsole = false;
    public static File logFile;
    public static final String hostName;
    
    static {
        InetAddress localHost = null;
        try {
            localHost = InetAddress.getLocalHost();
        } catch (Exception e) {
            //no-op
        }
        if (localHost != null) {
            hostName = localHost.getHostName();
        } else {
            hostName = "";
        }
    }

    /**
     * Logs the input string.
     * 
     * @param s the string to log
     */
    public static void log(String s) {
        s = "[INFO] (" + (hostName.equals("") ? "" : (hostName + ", ")) + new Date().toString() + "): " + s;
        if (logToConsole) {
            //log messages are just written to standard out-- for now
            System.out.println(s);
        }
        updateLogFile(s);
    }
    
    /**
     * Logs the input string as a warning.
     * 
     * @param s the string to log as a warning
     */
    public static void warn(String s) {
        s = "[WARN] (" + (hostName.equals("") ? "" : (hostName + ", ")) + new Date().toString() + "): " + s;
        if (logToConsole) {
            //log messages are just written to standard out-- for now
            System.out.println(s);
        }
        updateLogFile(s);
    }

    /**
     * Writes the input string to the error file/stream.
     * 
     * @param s the string to write to the error file/stream
     */
    public static void error(String s) {
        s = "[ERROR] (" + (hostName.equals("") ? "" : (hostName + ", ")) + new Date().toString() + "): " + s;
        if (logToConsole) {
            //error messages are just written to standard error-- for now
            System.err.println(s);
        }
        updateLogFile(s);
    }
    
    /**
     * Writes the stack trace for the input exception to the error file/stream
     *
     * @param ex the exception to log
     */
    public static void exception(Exception ex) {
        Logger.exception(ex, null);
    }

    /**
     * Writes the stack trace for the input exception to the error file/stream
     *
     * @param ex the exception to log
     * @param message the error message to write
     */
    public static void exception(Exception ex, String message) {
        if (logToConsole) {
            StringBuilder s = new StringBuilder();
            for (StackTraceElement ste : ex.getStackTrace()) {
                s.append(ste.toString());
                s.append(System.getProperty("line.separator"));
            }
	    if(message == null || message.isEmpty()) {
                Logger.error("An unexpected exception was encountered: ");
	    } else {
		Logger.error(message);
	    }
            Logger.error(s.toString());
        }
    }
    
    /**
     * Writes the input string to the log file.
     * 
     * @param s the string to write to the log file
     */
    public static void updateLogFile(String s) {
        try {
            BufferedWriter writer = new BufferedWriter(new FileWriter(logFile, true));
            writer.write(System.getProperty("line.separator"));
            writer.write(s);
            writer.flush();
            writer.close();
        } catch (Exception ex) {
            //no-op
        }
    }
}
