/**
 * Use of the WebDLPIndexer and related source code is subject to the
 * terms of the following license:
 *
 * Copyright (c) 2013 Carnegie Mellon University. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following acknowledgments and disclaimers.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. All advertising materials for third-party software mentioning features or
 * use of this software must display the following disclaimer:
 *
 * “Neither Carnegie Mellon University nor its Software Engineering Institute
 * have reviewed or endorsed this software”
 *
 * 4. The names “Carnegie Mellon University,” "CERT” and/or “Software
 * Engineering Institute" shall not be used to endorse or promote products
 * derived from this software without prior written permission. For written
 * permission, please contact permission@sei.cmu.edu.
 *
 * 5. Products derived from this software may not be called "CERT" nor may
 * "CERT" appear in their names without prior written permission of
 * permission@sei.cmu.edu.
 *
 * 6. Redistributions of any form whatsoever must retain the following
 * acknowledgment:
 *
 * "This product includes software developed by CERT with funding and support
 * from the US Government under Contract No. FA8721-05-C-0003. The U.S.
 * Government's rights to use, modify, reproduce, release, perform, display, or
 * disclose this material are restricted by the Rights in Technical
 * Data-Noncommercial Items clauses (DFAR 252-227.7013 and DFAR 252-227.7013
 * Alternate I contained in the foregoing identified contract.
 *
 * THIS SOFTWARE IS PROVIDED BY CARNEGIE MELLON UNIVERSITY ``AS IS'' AND
 * CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESS OR
 * IMPLIED, AS TO ANY MATTER, AND ALL SUCH WARRANTIES, INCLUDING WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, ARE EXPRESSLY
 * DISCLAIMED. WITHOUT LIMITING THE GENERALITY OF THE FOREGOING, CARNEGIE MELLON
 * UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND RELATING TO EXCLUSIVITY,
 * INFORMATIONAL CONTENT, ERROR-FREE OPERATION, RESULTS TO BE OBTAINED FROM USE,
 * FREEDOM FROM PATENT, TRADEMARK AND COPYRIGHT INFRINGEMENT AND/OR FREEDOM FROM
 * THEFT OF TRADE SECRETS.”
 *
 */
package webdlpindexer;

import java.io.File;
import java.io.IOException;
import java.net.*;
import org.apache.commons.io.FileUtils;
import webdlpindexer.indexer.IndexerHandler;

/**
 * A Server object continually listens for client requests and passes them on to
 * ClientHandler objects in their own thread. It also creates and starts the
 * IndexHandler object, which is the mediator for all index-related operations
 *
 * @author Todd Lewellen
 */
public class Server implements Runnable {

    private File indexDir;
    private File intellectualPropertyDir;
    private boolean newIndex;
    private boolean keepDelFiles;
    private boolean debug;
    private int portNum;
    private ServerSocket socket;
    IndexerHandler handler;

    /**
     *
     * @param indexDir The directory in which the index files are stored
     * @param intellectualPropertyDir The directory in which the intellectual
     * property files are stored
     * @param newIndex Whether or not a new index should be created from scratch
     * at the beginning of execution
     * @param keepDelFiles Whether or not files should be removed from the index
     * if they are deleted on the file system
     * @param debug Whether or not debugging is enabled (provides extra logging)
     * @param portNum The port on which the Server should listen for incoming
     * client requests
     */
    public Server(File indexDir, File intellectualPropertyDir, boolean newIndex,
            boolean keepDelFiles, boolean debug, int portNum) {

        this.indexDir = indexDir;
        this.intellectualPropertyDir = intellectualPropertyDir;
        this.newIndex = newIndex;
        this.keepDelFiles = keepDelFiles;
        this.debug = debug;
        this.portNum = portNum;

        //Create the tmp directory if it does not exist, otherwise cleans it
        File tmpDir = new File("tmp");
        if (!tmpDir.exists()) {
            tmpDir.mkdirs();
        } else {
            try {
                FileUtils.cleanDirectory(tmpDir);
            } catch (IOException ex) {
                Logger.exception(ex);
            }
        }
    }

    @Override
    public void run() {
        try {
            //Create ServerSocket and bind to localhost on portNum
            SocketAddress socketAddress = new InetSocketAddress(this.portNum);
            socket = new ServerSocket();
            socket.bind(socketAddress);
        } catch (BindException ex) {
            System.out.println("Error: Something is already running on port " + this.portNum + "!");
            System.out.println("Is another instance of the WebDLPIndexer running?");
            System.out.println("Exiting...");
            System.exit(1);
        } catch (IOException ex) {
            Logger.exception(ex);
        }

        handler = new IndexerHandler(indexDir, intellectualPropertyDir, newIndex, keepDelFiles, debug);
        Thread t = new Thread(handler);
        t.start();

        //Listen for communications
        while (true) {
            try {
                Socket clientSocket = socket.accept();
                ClientHandler clientHandler = new ClientHandler(handler, clientSocket);
                Thread clientHandlerThread = new Thread(clientHandler);
                clientHandlerThread.start();
            } catch (IOException ex) {
                Logger.exception(ex);
            }
        }
    }
}
