/**
 * Use of the WebDLPIndexer and related source code is subject to the
 * terms of the following license:
 *
 * Copyright (c) 2013 Carnegie Mellon University. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following acknowledgments and disclaimers.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. All advertising materials for third-party software mentioning features or
 * use of this software must display the following disclaimer:
 *
 * “Neither Carnegie Mellon University nor its Software Engineering Institute
 * have reviewed or endorsed this software”
 *
 * 4. The names “Carnegie Mellon University,” "CERT” and/or “Software
 * Engineering Institute" shall not be used to endorse or promote products
 * derived from this software without prior written permission. For written
 * permission, please contact permission@sei.cmu.edu.
 *
 * 5. Products derived from this software may not be called "CERT" nor may
 * "CERT" appear in their names without prior written permission of
 * permission@sei.cmu.edu.
 *
 * 6. Redistributions of any form whatsoever must retain the following
 * acknowledgment:
 *
 * "This product includes software developed by CERT with funding and support
 * from the US Government under Contract No. FA8721-05-C-0003. The U.S.
 * Government's rights to use, modify, reproduce, release, perform, display, or
 * disclose this material are restricted by the Rights in Technical
 * Data-Noncommercial Items clauses (DFAR 252-227.7013 and DFAR 252-227.7013
 * Alternate I contained in the foregoing identified contract.
 *
 * THIS SOFTWARE IS PROVIDED BY CARNEGIE MELLON UNIVERSITY ``AS IS'' AND
 * CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESS OR
 * IMPLIED, AS TO ANY MATTER, AND ALL SUCH WARRANTIES, INCLUDING WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, ARE EXPRESSLY
 * DISCLAIMED. WITHOUT LIMITING THE GENERALITY OF THE FOREGOING, CARNEGIE MELLON
 * UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND RELATING TO EXCLUSIVITY,
 * INFORMATIONAL CONTENT, ERROR-FREE OPERATION, RESULTS TO BE OBTAINED FROM USE,
 * FREEDOM FROM PATENT, TRADEMARK AND COPYRIGHT INFRINGEMENT AND/OR FREEDOM FROM
 * THEFT OF TRADE SECRETS.”
 *
 */
package webdlpindexer;

import java.io.File;

/**
 * This is the main class. This starts the application, parses input arguments,
 * and then starts the Server.
 *
 * @author Todd Lewellen
 */
public class WebDLPIndexer {

    /**
     * The default port upon which the server should listen if not otherwise
     * specified
     */
    public static final int DEFAULT_PORT = 22345;

    /**
     *
     * @param args Input arguments: indexDir, intellectualPropertyDir, newIndex,
     * keepDelFiles, debug, [server port]
     */
    public static void main(String[] args) {

        //Initialize the Logger
        String logDirName = "log";
        File logDir = new File(logDirName);
        if (!logDir.exists()) {
            logDir.mkdirs();
        }
        Logger.logFile = new File(logDirName + System.getProperty("file.separator") + "WebDLPIndexer.log");

        //Declare the arguments
        File indexDir = null;
        File intellectualPropertyDir = null;
        boolean newIndex = false;
        boolean keepDelFiles = false;
        boolean debug = false;
        int portNum = DEFAULT_PORT;

        //Parse and validate the two required arguments
        if (args.length >= 2) {
            indexDir = new File(args[0]);
            intellectualPropertyDir = new File(args[1]);
            String errorMessage = "";
            if (!indexDir.exists()) {
                errorMessage += "ERROR: Cannot find directory " + indexDir.toString() + "\n";
            }
            if (!intellectualPropertyDir.exists()) {
                errorMessage += "ERROR: Cannot find directory " + intellectualPropertyDir.toString() + "\n";
            }
            if (!indexDir.exists() || !intellectualPropertyDir.exists()) {
                Logger.error(errorMessage);
                System.out.println(errorMessage);
                System.out.println("Exiting...");
                System.exit(0);
            }
        } else {
            printHelp();
            System.out.println("Exiting...");
            System.exit(0);
        }
        //Parse any optional flags
        for (int i = 2; i < args.length; i++) {
            if (args[i].equals("--createIndex")) {
                newIndex = true;
            } else if (args[i].equals("--indexDeletedFiles")) {
                keepDelFiles = true;
            } else if (args[i].equals("--debug")) {
                debug = true;
            } else if (args[i].equals("--port")) {
                if (args[i + 1] != null) {
                    i++;
                    try {
                        portNum = Integer.parseInt(args[i]);
                    } catch (NumberFormatException ex) {
                        String errorMessage = "ERROR: " + args[i] + " does not appear to be a valid integer.\n"
                                + "Exiting...";
                        System.out.println(errorMessage);
                        Logger.error(errorMessage);
                        System.exit(0);
                    }
                } else {
                    String errorMessage = "ERROR: No port number specified after flag '--port'\n"
                                        + "Exiting...";
                    System.out.println(errorMessage);
                    Logger.error(errorMessage);
                    System.exit(0);
                }
            } else {
                String errorMessage = "ERROR: Unknown argument '" + args[i] +"'\n";
                Logger.error(errorMessage);
                System.out.println(errorMessage);
                printHelp();
                System.exit(0);
            }
        }

        if (debug) {
            System.out.println("Intellectual Property directory : " + intellectualPropertyDir);
            System.out.println("Index directory:                : " + indexDir);
            System.out.println("Create new index upon startup   : " + newIndex);
            System.out.println("Keep deleted files in index     : " + keepDelFiles);
            System.out.println("Debug Mode                      : " + debug);
            System.out.println("Port                            : " + portNum);
        }

        //Start the Server
        Server server = new Server(indexDir, intellectualPropertyDir, newIndex, keepDelFiles, debug, portNum);
        Thread t = new Thread(server);
        t.start();
    }

    private static void printHelp() {
        System.out.println("The following arguments are required:");
        System.out.println("  arg 1: Path of intellectual property directory");
        System.out.println("  arg 2: Path of existing or desired index directory");
        System.out.println("  arg 3: (true/false) Whether or not the index should be built from scratch");
        System.out.println("  arg 4: (true/false) Whether or not deleted intellectual property files should be kept in the index");
        System.out.println("  arg 5: (true/false) Enable debug mode");
        System.out.println("  (optional) arg 6 : TCP port number for server");
        System.out.println("");
    }
}
