/**
 * Use of the WebDLPIndexer and related source code is subject to the
 * terms of the following license:
 *
 * Copyright (c) 2013 Carnegie Mellon University. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following acknowledgments and disclaimers.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. All advertising materials for third-party software mentioning features or
 * use of this software must display the following disclaimer:
 *
 * “Neither Carnegie Mellon University nor its Software Engineering Institute
 * have reviewed or endorsed this software”
 *
 * 4. The names “Carnegie Mellon University,” "CERT” and/or “Software
 * Engineering Institute" shall not be used to endorse or promote products
 * derived from this software without prior written permission. For written
 * permission, please contact permission@sei.cmu.edu.
 *
 * 5. Products derived from this software may not be called "CERT" nor may
 * "CERT" appear in their names without prior written permission of
 * permission@sei.cmu.edu.
 *
 * 6. Redistributions of any form whatsoever must retain the following
 * acknowledgment:
 *
 * "This product includes software developed by CERT with funding and support
 * from the US Government under Contract No. FA8721-05-C-0003. The U.S.
 * Government's rights to use, modify, reproduce, release, perform, display, or
 * disclose this material are restricted by the Rights in Technical
 * Data-Noncommercial Items clauses (DFAR 252-227.7013 and DFAR 252-227.7013
 * Alternate I contained in the foregoing identified contract.
 *
 * THIS SOFTWARE IS PROVIDED BY CARNEGIE MELLON UNIVERSITY ``AS IS'' AND
 * CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESS OR
 * IMPLIED, AS TO ANY MATTER, AND ALL SUCH WARRANTIES, INCLUDING WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, ARE EXPRESSLY
 * DISCLAIMED. WITHOUT LIMITING THE GENERALITY OF THE FOREGOING, CARNEGIE MELLON
 * UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND RELATING TO EXCLUSIVITY,
 * INFORMATIONAL CONTENT, ERROR-FREE OPERATION, RESULTS TO BE OBTAINED FROM USE,
 * FREEDOM FROM PATENT, TRADEMARK AND COPYRIGHT INFRINGEMENT AND/OR FREEDOM FROM
 * THEFT OF TRADE SECRETS.”
 *
 */
package webdlpindexer.indexer;

import java.io.File;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.net.Socket;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import org.apache.commons.io.monitor.FileAlterationListener;
import org.apache.commons.io.monitor.FileAlterationMonitor;
import org.apache.commons.io.monitor.FileAlterationObserver;
import webdlpindexer.Logger;
import webdlpindexer.indexer.indexrequest.ClientIndexRequest;
import webdlpindexer.indexer.indexrequest.FilesystemIndexRequest;
import webdlpindexer.indexer.indexrequest.IndexRequest;

/**
 * This class acts as a manager of the Lucene index.  It takes requests from clients and the file system, 
 * adds them to its queue of requests, and processes them from the queue in a FIFO fashion.
 * 
 * @author Todd Lewellen
 */
public class IndexerHandler implements Runnable {

    private File indexDir;
    private File intellectualPropertyDir;
    private boolean newIndex;
    private boolean keepDelFiles;
    private boolean debug;
    private Indexer indexer;
    private BlockingQueue queue;

    /**
     * @param indexDir The directory in which the index files are stored
     * @param intellectualPropertyDir The directory in which the intellectual property files are stored
     * @param newIndex Whether or not a new index should be created from scratch at the beginning of execution
     * @param keepDelFiles Whether or not files should be removed from the index if they are deleted on the file system
     * @param debug Whether or not debugging is enabled (provides extra logging)
     */
    public IndexerHandler(File indexDir, File intellectualPropertyDir, boolean newIndex, boolean keepDelFiles, boolean debug) {
        this.indexDir = indexDir;
        this.intellectualPropertyDir = intellectualPropertyDir;
        this.newIndex = newIndex;
        this.keepDelFiles = keepDelFiles;
        this.debug = debug;
        queue = new LinkedBlockingQueue();
    }

    @Override
    public void run() {
        //If the indexDir does not exist, create it
        if(!indexDir.exists())
            indexDir.mkdir();

        //Create the indexer
        indexer = new Indexer(indexDir, intellectualPropertyDir, keepDelFiles, debug);
        if (newIndex) {
            indexer.indexAll();
        }

        if(debug){
            String message = "Now monitoring directory " + intellectualPropertyDir.getAbsolutePath() + " for changes.";
            System.out.println(message);
            Logger.log(message);
        }
        //Listen for changes to the filesystem in the intellectualPropertyDir
        FileAlterationObserver observer = new FileAlterationObserver(intellectualPropertyDir);
        observer.addListener(new FileListener());   //FileListener defined as an inner class below
        long interval = 1000;                       //Check every 1000 ms
        FileAlterationMonitor monitor = new FileAlterationMonitor(interval);
        monitor.addObserver(observer);
        try {
            monitor.start();
        } catch (Exception ex) {
            Logger.exception(ex);
            String error = "ERROR! Unable to monitor the intellectual property directory @ " 
                            + intellectualPropertyDir.toString() + "\n" + "System exiting...";
            Logger.error(error);
            System.out.println(error);
            System.exit(0);
        }

        IndexRequest request;
        boolean loop = true;    //Right now, loop is always true.  Provided for future functionality 
                                //in case certain conditions require the application to exit (something can set loop=false)

        //Block on the queue and process requests as they come in
        while (loop) {
            try {
                request = (IndexRequest) queue.take();
                File file = request.getFile();

                /*
                 * If ClientIndexRequest
                 */
                if (request.getClass() == ClientIndexRequest.class) {
                    ClientIndexRequest clientRequest = (ClientIndexRequest) request;
                    ClientIndexRequest.RequestType requestType = clientRequest.getRequestType();
                    try {
                        Socket clientSocket = clientRequest.getSocket();
                        ObjectOutputStream writer = new ObjectOutputStream(clientSocket.getOutputStream());
                        switch (requestType) {
                            case TEST_SIMILARITY:
                                writer.writeObject(indexer.testSimilarity(file));
                                break;
                            case TEST_SIMILARITY_WITH_THRESHOLD:
                                writer.writeObject(indexer.testSimilarityWithThreshold(file, clientRequest.getThreshold()));
                                break;
                        }
                        writer.flush();
                        writer.close();
                        clientSocket.close();
                        
                        //Remove the temp file created by the request from the index
                        indexer.forceRemoveDocument(file);
                        //Remove the temp file create by the request from the filesystem
                        if(!debug)                          
                            request.getFile().delete();                                                                           
                        if (debug)
                            Logger.log(requestType.toString() + ": " + request.getFile().getName());                        
                    } catch (IOException ex) {
                        Logger.exception(ex);
                    }
                    /*
                     * If FilesystemIndexRequest
                     */
                } else if (request.getClass() == FilesystemIndexRequest.class) {
                    FilesystemIndexRequest systemRequest = (FilesystemIndexRequest) request;
                    FilesystemIndexRequest.RequestType requestType = systemRequest.getRequestType();
                    switch (requestType) {
                        case ADD_FILE:
                            indexer.indexDocument(file);
                            break;
                        case MODIFY_FILE:
                            indexer.modifyDocument(file);
                            break;
                        case REMOVE_FILE:
                            indexer.removeDocument(file);
                            break;
                    }
                    if (debug) {
                        Logger.log(requestType + ": " + file.getName());
                        Logger.log(indexer.getIndexInfo());
                    }
                }
                
            } catch (InterruptedException ex) {
                Logger.exception(ex);
            }
            if(debug)
                Logger.log("Queuing " + queue.size() + " requests...");
        }
    }

    /**
     * 
     * @param request The IndexRequest object to be processed
     */
    public void submitIndexRequest(IndexRequest request) {
        queue.add(request);
    }

    // A private inner Listener class for responding to file system notifications
    private class FileListener implements FileAlterationListener {

        @Override
        public void onStart(FileAlterationObserver fao) {
        }

        @Override
        public void onDirectoryCreate(File file) {
            throw new UnsupportedOperationException("onDirectoryCreate not supported yet.");
        }

        @Override
        public void onDirectoryChange(File file) {
            throw new UnsupportedOperationException("onDirectoryChange not supported yet.");
        }

        @Override
        public void onDirectoryDelete(File file) {
            throw new UnsupportedOperationException("onDirectoryDelete not supported yet.");
        }

        @Override
        public void onFileCreate(File file) {
            submitIndexRequest(new FilesystemIndexRequest(FilesystemIndexRequest.RequestType.ADD_FILE, file));            
        }

        @Override
        public void onFileChange(File file) {
            submitIndexRequest(new FilesystemIndexRequest(FilesystemIndexRequest.RequestType.MODIFY_FILE, file));           
        }

        @Override
        public void onFileDelete(File file) {            
                submitIndexRequest(new FilesystemIndexRequest(FilesystemIndexRequest.RequestType.REMOVE_FILE, file));            
        }

        @Override
        public void onStop(FileAlterationObserver fao) {
        }
    }
}
