/**
 * Copyright (c) 2004-2023 Carnegie Mellon University and others.
 * (see Contributors file).
 *
 * All Rights Reserved.
 *
 * NO WARRANTY. ALL MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
 * UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED, AS
 * TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR PURPOSE
 * OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE
 * MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND
 * WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Created, in part, with funding and support from the United States Government.
 * (see Acknowledgments file).
 *
 * This program includes and/or can make use of certain third party source code,
 * object code, documentation and other files ("Third Party Software"). The
 * Third Party Software that is used by this program is dependent upon your
 * system configuration. By using this program, You agree to comply with any and
 * all relevant Third Party Software terms and conditions contained in any such
 * Third Party Software or separate license file distributed with such Third
 * Party Software. The parties who own the Third Party Software ("Third Party
 * Licensors") are intended third party beneficiaries to this License with
 * respect to the terms applicable to their Third Party Software. Third Party
 * Software licenses only apply to the Third Party Software and not any other
 * portion of this program or this program as a whole.
 */
package org.osate.isse.verificationmethods;

import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import org.eclipse.core.runtime.ILog;
import org.eclipse.core.runtime.Platform;
import org.osate.aadl2.instance.ComponentInstance;
import org.osgi.framework.Bundle;
import org.sireum.aadl.osate.util.Util;
import org.sireum.awas.AADLBridge.AadlHandler;
import org.sireum.awas.ast.Model;
import org.sireum.awas.awasfacade.AwasGraphImpl;
import org.sireum.awas.awasfacade.Collector;
import org.sireum.awas.flow.FlowEdge;
import org.sireum.awas.flow.FlowGraph;
import org.sireum.awas.flow.FlowNode;
import org.sireum.awas.symbol.SymbolTable;
import org.sireum.util.ConsoleTagReporter;

public class AwasMethods {
	private static final String BUNDLE_NAME;
	private static final ILog LOG;
	private AwasGraphImpl awasGraph;
	private FlowGraph<FlowNode, FlowEdge<FlowNode>> flowGraph;

	static {
		Bundle bundle = Activator.getContext().getBundle();
		BUNDLE_NAME = bundle.getSymbolicName();
		LOG = Platform.getLog(bundle);
	}

	public AwasMethods(ComponentInstance component) {
		try {
			Model awasModel = AadlHandler.buildAwasModel(Util.getAir(component));
			SymbolTable symbolTable = SymbolTable.apply(awasModel, new ConsoleTagReporter());
			flowGraph = FlowGraph.apply(awasModel, symbolTable, false);
			awasGraph = new AwasGraphImpl(flowGraph, symbolTable);
		} catch (Throwable e) {
			e.printStackTrace();
		}
	}

	public String getDot() {
		return awasGraph.getDotGraph();
	}

	public int getNumEdges() {
		return flowGraph.numOfEdges();
	}

	public int getNumNodes() {
		return flowGraph.numOfNodes();
	}

	public void runQuery(String query) {
		try {
			ExecutorService executor = Executors.newSingleThreadExecutor();
			Callable<Map<String, Collector>> qryRunner = new runLongQuery(query);
			Map<String, Collector> queryRes = executor.submit(qryRunner).get(30, TimeUnit.SECONDS);
			queryRes = null; // dummy statement, inspect the results here if you want
		} catch (TimeoutException e) {
			// Don't do anything, this is fine
		} catch (Throwable e) {
			e.printStackTrace();
		}
	}

	private class runLongQuery implements Callable<Map<String, Collector>> {
		String query;

		public runLongQuery(String query) {
			this.query = query;
		}

	    @Override
		public Map<String, Collector> call() {
	        	Map<String, Collector> queryRes = null;
				try {
					queryRes = (awasGraph).queryEvaluator(query);
				} catch (Exception e) {
					e.printStackTrace();
				}
				return queryRes;
	        }
	    }
}