/*******************************************************************************
 * Copyright (c) 2004-2022 Carnegie Mellon University and others. (see Contributors file).
 * All Rights Reserved.
 *
 * NO WARRANTY. ALL MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
 * KIND, EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR PURPOSE
 * OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
 * MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
 *
 * This program and the accompanying materials are made available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 * SPDX-License-Identifier: EPL-2.0
 *
 * Created, in part, with funding and support from the United States Government. (see Acknowledgments file).
 *
 * This program includes and/or can make use of certain third party source code, object code, documentation and other
 * files ("Third Party Software"). The Third Party Software that is used by this program is dependent upon your system
 * configuration. By using this program, You agree to comply with any and all relevant Third Party Software terms and
 * conditions contained in any such Third Party Software or separate license file distributed with such Third Party
 * Software. The parties who own the Third Party Software ("Third Party Licensors") are intended third party beneficiaries
 * to this license with respect to the terms applicable to their Third Party Software. Third Party Software licenses
 * only apply to the Third Party Software and not any other portion of this program or this program as a whole.
 *******************************************************************************/
package org.osate.slicer.evaluation;

import static org.junit.Assert.assertTrue;

import java.net.URI;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.time.Duration;
import java.time.Instant;
import java.util.stream.Collectors;

import org.eclipse.xtext.testing.InjectWith;
import org.eclipse.xtext.testing.XtextRunner;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.osate.aadl2.AadlPackage;
import org.osate.aadl2.SystemImplementation;
import org.osate.aadl2.errormodel.instance.instantiator.EMV2AnnexInstantiator;
import org.osate.aadl2.instance.SystemInstance;
import org.osate.aadl2.instantiation.InstantiateModel;
import org.osate.slicer.SlicerRepresentation;
import org.osate.testsupport.Aadl2InjectorProvider;
import org.osate.testsupport.TestHelper;

import com.google.inject.Inject;

@RunWith(XtextRunner.class)
@InjectWith(Aadl2InjectorProvider.class)
public class DisplayManager {

	@Inject
	TestHelper<AadlPackage> myTestHelper;

	private SlicerRepresentation tlg;
	private SystemInstance si;

	private int vertexCount, edgeCount;
	private Duration instantiationTime, graphGen1Time, graphGen2Time, graphGen3Time, forwardReach1Time,
			forwardReach2Time, forwardReach3Time, backwardReach1Time, backwardReach2Time, backwardReach3Time,
			reachFrom1Time, reachFrom2Time, reachFrom3Time, reachThrough1Time, reachThrough2Time, reachThrough3Time;

	private final String FORWARD_REACH_ORIGIN1 = "display_manager_i_Instance.infos.altitude_processing.sensed_value";
	private final String FORWARD_REACH_ORIGIN2 = "display_manager_i_Instance.manager.critical.altitude";
	private final String FORWARD_REACH_ORIGIN3 = "display_manager_i_Instance.infos.gps_sensor.pos";

	private final String BACKWARD_REACH_ORIGIN1 = "display_manager_i_Instance.renderer.page_warning";
	private final String BACKWARD_REACH_ORIGIN2 = "display_manager_i_Instance.manager.altitude";
	private final String BACKWARD_REACH_ORIGIN3 = "display_manager_i_Instance.infos.wind_processing.speed";

	private final String REACH_FROM_ORIGIN1 = "display_manager_i_Instance.monitor.fuels.level1";
	private final String REACH_FROM_TARGET1 = "display_manager_i_Instance.manager.page_info";
	private final String REACH_FROM_ORIGIN2 = "display_manager_i_Instance.infos.gps_sensor.pos";
	private final String REACH_FROM_TARGET2 = "display_manager_i_Instance.renderer.warning.page";
	private final String REACH_FROM_ORIGIN3 = "display_manager_i_Instance.renderer.page_critical";
	private final String REACH_FROM_TARGET3 = "display_manager_i_Instance.infos.wind_processing.sensed";

	private final String REACH_THROUGH_ORIGIN1 = "display_manager_i_Instance.monitor.fuels.level1";
	private final String REACH_THROUGH_MID1 = "display_manager_i_Instance.manager.infos.fuel_status";
	private final String REACH_THROUGH_TARGET1 = "display_manager_i_Instance.manager.page_info";
	private final String REACH_THROUGH_ORIGIN2 = "display_manager_i_Instance.infos.gps_sensor.pos";
	private final String REACH_THROUGH_MID2 = "display_manager_i_Instance.manager.warning.aircraft_speed";
	private final String REACH_THROUGH_TARGET2 = "display_manager_i_Instance.renderer.warning.page";
	private final String REACH_THROUGH_ORIGIN3 = "display_manager_i_Instance.renderer.page_critical";
	private final String REACH_THROUGH_MID3 = "display_manager_i_Instance.manager.critical.pitot_status";
	private final String REACH_THROUGH_TARGET3 = "display_manager_i_Instance.infos.wind_processing.sensed";

	private final String TOP_DIRECTORY = "data/aadl-projects/display-manager/initial-model/";
	private final String MAIN_FILE_NAME = "integration.aadl";
	private final String SYS_IMPL_NAME = "display_manager.i";

	@Before
	public void setUp() throws Exception {
		System.setProperty(EMV2AnnexInstantiator.PROPERTY_NAME, "true");
		tlg = new SlicerRepresentation();

		var mainFile = TOP_DIRECTORY + MAIN_FILE_NAME;
		var supportingFiles = Files
				.find(Paths.get(new URI("file://" + System.getProperty("user.dir") + "/../" + TOP_DIRECTORY)),
				Integer.MAX_VALUE, (filePath, fileAttr) -> fileAttr.isRegularFile()
								&& filePath.toString().toLowerCase().endsWith(".aadl")
								&& !filePath.toString().endsWith(MAIN_FILE_NAME))
				.map(path -> path.toString().substring(path.toString().indexOf(TOP_DIRECTORY)))
				.collect(Collectors.toList())
				.toArray(String[]::new);
		var pkg = myTestHelper.parseFile(mainFile, supportingFiles);
		var impl = (SystemImplementation) pkg.getPublicSection()
				.getOwnedClassifiers()
				.stream()
				.filter(c -> c.getName().equals(SYS_IMPL_NAME))
				.findFirst()
				.get();

		var instantiationStart = Instant.now();
		si = InstantiateModel.instantiate(impl);
		var instantiationEnd = Instant.now();
		var graphGen1Start = Instant.now();
		tlg.buildGraph(si);
		var graphGen1End = Instant.now();
		tlg = new SlicerRepresentation();
		var graphGen2Start = Instant.now();
		tlg.buildGraph(si);
		var graphGen2End = Instant.now();
		tlg = new SlicerRepresentation();
		var graphGen3Start = Instant.now();
		tlg.buildGraph(si);
		var graphGen3End = Instant.now();

		vertexCount = tlg.getGraph().vertexSet().size();
		edgeCount = tlg.getGraph().edgeSet().size();
		instantiationTime = Duration.between(instantiationStart, instantiationEnd);
		graphGen1Time = Duration.between(graphGen1Start, graphGen1End);
		graphGen2Time = Duration.between(graphGen2Start, graphGen2End);
		graphGen3Time = Duration.between(graphGen3Start, graphGen3End);
	}

	@After
	public void tearDown() throws Exception {
		var data = new PerformanceData("DisplayManager", vertexCount, edgeCount, instantiationTime, graphGen1Time,
				graphGen2Time, graphGen3Time, forwardReach1Time, forwardReach2Time, forwardReach3Time,
				backwardReach1Time, backwardReach2Time, backwardReach3Time, reachFrom1Time, reachFrom2Time,
				reachFrom3Time, reachThrough1Time, reachThrough2Time, reachThrough3Time);
		// System.out.println(tlg.toDot(tlg.getGraph()));
		System.out.println(data.toCSV());
	}

	@Test
	public void queries() {
		var g = tlg.getGraph();
		var rg = tlg.getReversedGraph();

		var forwardReach1Start = Instant.now();
		tlg.reach(g, FORWARD_REACH_ORIGIN1);
		var forwardReach1End = Instant.now();
		var forwardReach2Start = Instant.now();
		tlg.reach(g, FORWARD_REACH_ORIGIN2);
		var forwardReach2End = Instant.now();
		var forwardReach3Start = Instant.now();
		tlg.reach(g, FORWARD_REACH_ORIGIN3);
		var forwardReach3End = Instant.now();
		forwardReach1Time = Duration.between(forwardReach1Start, forwardReach1End);
		forwardReach2Time = Duration.between(forwardReach2Start, forwardReach2End);
		forwardReach3Time = Duration.between(forwardReach3Start, forwardReach3End);

		var backwardReach1Start = Instant.now();
		tlg.reach(rg, BACKWARD_REACH_ORIGIN1);
		var backwardReach1End = Instant.now();
		var backwardReach2Start = Instant.now();
		tlg.reach(rg, BACKWARD_REACH_ORIGIN2);
		var backwardReach2End = Instant.now();
		var backwardReach3Start = Instant.now();
		tlg.reach(rg, BACKWARD_REACH_ORIGIN3);
		var backwardReach3End = Instant.now();
		backwardReach1Time = Duration.between(backwardReach1Start, backwardReach1End);
		backwardReach2Time = Duration.between(backwardReach2Start, backwardReach2End);
		backwardReach3Time = Duration.between(backwardReach3Start, backwardReach3End);

		var reachFrom1Start = Instant.now();
		tlg.reachFrom(g, REACH_FROM_ORIGIN1, REACH_FROM_TARGET1);
		var reachFrom1End = Instant.now();
		var reachFrom2Start = Instant.now();
		tlg.reachFrom(g, REACH_FROM_ORIGIN2, REACH_FROM_TARGET2);
		var reachFrom2End = Instant.now();
		var reachFrom3Start = Instant.now();
		tlg.reachFrom(g, REACH_FROM_ORIGIN3, REACH_FROM_TARGET3);
		var reachFrom3End = Instant.now();
		reachFrom1Time = Duration.between(reachFrom1Start, reachFrom1End);
		reachFrom2Time = Duration.between(reachFrom2Start, reachFrom2End);
		reachFrom3Time = Duration.between(reachFrom3Start, reachFrom3End);

		var reachThrough1Start = Instant.now();
		tlg.reachThrough(g, REACH_THROUGH_ORIGIN1, REACH_THROUGH_MID1, REACH_THROUGH_TARGET1, false);
		var reachThrough1End = Instant.now();
		var reachThrough2Start = Instant.now();
		tlg.reachThrough(g, REACH_THROUGH_ORIGIN2, REACH_THROUGH_MID2, REACH_THROUGH_TARGET2, false);
		var reachThrough2End = Instant.now();
		var reachThrough3Start = Instant.now();
		tlg.reachThrough(g, REACH_THROUGH_ORIGIN3, REACH_THROUGH_MID3, REACH_THROUGH_TARGET3, false);
		var reachThrough3End = Instant.now();
		reachThrough1Time = Duration.between(reachThrough1Start, reachThrough1End);
		reachThrough2Time = Duration.between(reachThrough2Start, reachThrough2End);
		reachThrough3Time = Duration.between(reachThrough3Start, reachThrough3End);

		assertTrue(true);
	}
}